#include "client.h"
#include "sessionmanager.h"
#include "proto.h"
#include "debug.h"


Client::Client(int fd) : TCPStream(fd)
{
	numSessions++;
}

Client::~Client()
{
	while (!sessionList.isEmpty()) {
		Session *s = LIST_ENTRY(sessionList.removeHead(), Session, listItem);
		s->hashItem.remove();
		delete s;
	}
}

void Client::deliver(const char *to, const char *from, uint16 cmd, const char *data, int n)
{
	OutPacket out;
	out << to << from << cmd;

	sendPacket(out, data, n);
}

void Client::onAddSession(InPacket &in)
{
	ICQ_STR name;
	in >> name;

	Session *s = new Session(this, name.text);
	s->onChangeStatus(in);
	s->onUpdateUserInfo(in);

	sessionList.add(&s->listItem);
	sessionManager.addSession(s);

	numSessions++;
}

void Client::onRemoveSession(InPacket &in)
{
	ICQ_STR name;
	in >> name;

	Session *s = sessionManager.getSession(name.text);
	if (!s) {
		ICQ_LOG("No session %s to remove\n", name.text);
		return;
	}

	s->hashItem.remove();
	s->listItem.remove();
	delete s;

	numSessions--;
}

void Client::onDeliver(InPacket &in)
{
	uint8 online;
	ICQ_STR to, from;
	uint16 cmd;

	in >> to >> from >> online >> cmd;

	Client *client = NULL;
	Session *s = sessionManager.getSession(to.text);

	if (s)
		client = s->client;
	else if (!online) {
		ICQ_LOG("%s is offline, bounce it\n", to.text);
		client = (numSessions ? this : sessionManager.chooseClient());
	}

	if (client)
		client->deliver(to.text, from.text, cmd, in.cursor, in.getBytesLeft());
}

void Client::onRequest(InPacket &in)
{
	ICQ_STR from;
	in >> from;

	Session *s = sessionManager.getSession(from.text);
	if (s)
		s->handleRequest(in);
}

void Client::onPacketReceived(const char *data, int n)
{
	InPacket in(data, n);

	uint16 cmd;
	in >> cmd;

	switch (cmd) {
	case SM_ADD_SESSION:
		onAddSession(in);
		break;

	case SM_REMOVE_SESSION:
		onRemoveSession(in);
		break;

	case SM_DELIVER:
		onDeliver(in);
		break;

	case SM_REQUEST:
		onRequest(in);
		break;

	default:
		ICQ_LOG("Unknown sm cmd\n");
	}
}

void Client::onSocketClose()
{
	TCPStream::onSocketClose();

	sessionManager.removeClient(this);
}
